<?php

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit;


/**
 * List table class outputter for Calendars
 *
 */
Class WPSBC_WP_List_Table_Legend_Items extends WPSBC_WP_List_Table {

	/**
	 * The number of calendars that should appear in the table
	 *
	 * @access private
	 * @var int
	 *
	 */
	private $items_per_page;

	/**
	 * The number of the page being displayed by the pagination
	 *
	 * @access private
	 * @var int
	 *
	 */
	private $paged;

	/**
	 * The data of the table
	 *
	 * @access public
	 * @var array
	 *
	 */
	public $data = array();


	/**
	 * Constructor
	 *
	 */
	public function __construct() {

		parent::__construct( array(
			'plural' 	=> 'wpsbc_legend_items',
			'singular' 	=> 'wpsbc_legend_item',
			'ajax' 		=> false
		));

		$this->items_per_page = 500;
		$this->paged 		  = ( ! empty( $_GET['paged'] ) ? (int)$_GET['paged'] : 1 );

		// Get and set table data
		$this->set_table_data();
		
		// Add column headers and table items
		$this->_column_headers = array( $this->get_columns(), array(), array() );
		$this->items 		   = $this->data;

	}


	/**
	 * Returns all the columns for the table
	 *
	 */
	public function get_columns() {

		$columns = array(
			'sort'		 => __( 'Sort', 'wp-simple-booking-calendar' ),
			'color'  	 => __( 'Color', 'wp-simple-booking-calendar' ),
			'name'		 => __( 'Name', 'wp-simple-booking-calendar' ),
			'is_default' => __( 'Default', 'wp-simple-booking-calendar' ),
			'is_visible' => __( 'Visible', 'wp-simple-booking-calendar' ),
			'is_bookable' => __('Bookable', 'wp-simple-booking-calendar') . ' ' . wpsbc_get_output_tooltip(__('Choose wether or not a legend item is Bookable. The search widget will only return Bookable legend items as available dates.', 'wp-simple-booking-calendar')),
		);

		/**
		 * Filter the columns of the legend items table
		 *
		 * @param array $columns
		 *
		 */
		return apply_filters( 'wpsbc_list_table_calendars_columns', $columns );

	}


	/**
	 * Gets the legend items data and sets it
	 *
	 */
	private function set_table_data() {

		$legend_items_args  = array(
			'calendar_id' => ( ! empty( $_GET['calendar_id'] ) ? absint( $_GET['calendar_id'] ) : 0 )
		);

		$legend_items = wpsbc_get_legend_items( $legend_items_args );

		if( empty( $legend_items ) )
			return;

		foreach( $legend_items as $legend_item ) {

			$row_data = $legend_item->to_array();

			/**
			 * Filter the legend item row data
			 *
			 * @param array 		    $row_data
			 * @param WPSBC_Legend_Item $legend_item
			 *
			 */
			$row_data = apply_filters( 'wpsbc_list_table_legend_items_row_data', $row_data, $legend_item );

			$this->data[] = $row_data;

		}
		
	}


	/**
	 * Returns the HTML that will be displayed in each columns
	 *
	 * @param array $item 			- data for the current row
	 * @param string $column_name 	- name of the current column
	 *
	 * @return string
	 *
	 */
	public function column_default( $item, $column_name ) {

		return isset( $item[ $column_name ] ) ? $item[ $column_name ] : '-';

	}


	/**
	 * Returns the HTML that will be displayed in the "sort" column
	 *
	 * @param array $item - data for the current row
	 *
	 * @return string
	 *
	 */
	public function column_sort( $item ) {

		return '<span data-id="' . $item['id'] . '" class="wpsbc-move-legend-item"><span class="wpsbc-inner"></span></span>';

	}


	/**
	 * Returns the HTML that will be displayed in the "color" column
	 *
	 * @param array $item - data for the current row
	 *
	 * @return string
	 *
	 */
	public function column_color( $item ) {

		$output = wpsbc_get_legend_item_icon( $item['id'], $item['type'], $item['color'] );

		return $output;

	}


	/**
	 * Returns the HTML that will be displayed in the "name" column
	 *
	 * @param array $item - data for the current row
	 *
	 * @return string
	 *
	 */
	public function column_name( $item ) {

		$output  = '<strong><a class="row-title" href="' . add_query_arg( array( 'page' => 'wpsbc-calendars', 'subpage' => 'edit-legend-item', 'legend_item_id' => $item['id'], 'calendar_id' => $item['calendar_id'] ) , admin_url( 'admin.php' ) ) . '">' . ( !empty( $item['name'] ) ? $item['name'] : '' ) . '</a></strong>';

		$actions = array();

		$actions['edit'] = '<a href="' . add_query_arg( array( 'page' => 'wpsbc-calendars', 'subpage' => 'edit-legend-item', 'legend_item_id' => $item['id'], 'calendar_id' => $item['calendar_id'] ) , admin_url( 'admin.php' ) ) . '">' . __( 'Edit', 'wp-simple-booking-calendar' ) . '</a>';

		if( empty( $item['is_default'] ) )
			$actions['delete'] = '<span class="trash"><a onclick="return confirm( \'' . __( "Are you sure you want to delete this legend item?", "wp-simple-booking-calendar" ) . ' \' )" href="' . wp_nonce_url( add_query_arg( array( 'page' => 'wpsbc-calendars', 'subpage' => 'view-legend', 'wpsbc_action' => 'delete_legend_item', 'legend_item_id' => $item['id'] ) , admin_url( 'admin.php' ) ), 'wpsbc_delete_legend_item', 'wpsbc_token' ) . '" class="submitdelete">' . __( 'Delete', 'wp-simple-booking-calendar' ) . '</a></span>';

		$output .= $this->row_actions( $actions );

		return $output;

	}


	/**
	 * Returns the HTML that will be displayed in the "is_default" column
	 *
	 * @param array $item - data for the current row
	 *
	 * @return string
	 *
	 */
	public function column_is_default( $item ) {

		$output = '<span class="wpsbc-list-table-icon wpsbc-list-table-icon-star ' . ( ! empty( $item['is_default'] ) ? 'wpsbc-list-table-icon-active' : 'wpsbc-list-table-icon-inactive' ) . '">';

		if( ! empty( $item['is_default'] ) ) {

			$output .= '<span class="dashicons dashicons-star-filled"></span>';

		} else {

			$output .= '<a href="' . ( wp_nonce_url( add_query_arg( array( 'wpsbc_action' => 'make_default_legend_item', 'legend_item_id' => $item['id'] ), remove_query_arg( 'wpsbc_message' ) ), 'wpsbc_make_default_legend_item', 'wpsbc_token' ) ) . '"><span class="dashicons dashicons-star-filled"></span></a>';

		}

		$output .= '</span>';

		return $output;

	}


	/**
	 * Returns the HTML that will be displayed in the "is_visible" column
	 *
	 * @param array $item - data for the current row
	 *
	 * @return string
	 *
	 */
	public function column_is_visible( $item ) {

		$output = '<span class="wpsbc-list-table-icon wpsbc-list-table-icon-yes ' . ( ! empty( $item['is_visible'] ) ? 'wpsbc-list-table-icon-active' : 'wpsbc-list-table-icon-inactive' ) . '">';

		if( ! empty( $item['is_visible'] ) ) {

			$output .= '<a href="' . ( wp_nonce_url( add_query_arg( array( 'wpsbc_action' => 'make_invisible_legend_item', 'legend_item_id' => $item['id'] ), remove_query_arg( 'wpsbc_message' ) ), 'wpsbc_make_invisible_legend_item', 'wpsbc_token' ) ) . '"><span class="dashicons dashicons-yes"></span></a>';

		} else {

			$output .= '<a href="' . ( wp_nonce_url( add_query_arg( array( 'wpsbc_action' => 'make_visible_legend_item', 'legend_item_id' => $item['id'] ), remove_query_arg( 'wpsbc_message' ) ), 'wpsbc_make_visible_legend_item', 'wpsbc_token' ) ) . '"><span class="dashicons dashicons-yes"></span></a>';

		}

		$output .= '</span>';

		return $output;

	}

	/**
     * Returns the HTML that will be displayed in the "is_bookable" column
     *
     * @param array $item - data for the current row
     *
     * @return string
     *
     */
    public function column_is_bookable($item)
    {

        $output = '<span class="wpsbc-list-table-icon wpsbc-list-table-icon-yes ' . (!empty($item['is_bookable']) ? 'wpsbc-list-table-icon-active' : 'wpsbc-list-table-icon-inactive') . '">';

        if (!empty($item['is_bookable'])) {

            $output .= '<a href="' . (wp_nonce_url(add_query_arg(array('wpsbc_action' => 'make_unbookable_legend_item', 'legend_item_id' => $item['id']), remove_query_arg('wpsbc_message')), 'wpsbc_make_unbookable_legend_item', 'wpsbc_token')) . '"><span class="dashicons dashicons-yes"></span></a>';

        } else {

            $output .= '<a href="' . (wp_nonce_url(add_query_arg(array('wpsbc_action' => 'make_bookable_legend_item', 'legend_item_id' => $item['id']), remove_query_arg('wpsbc_message')), 'wpsbc_make_bookable_legend_item', 'wpsbc_token')) . '"><span class="dashicons dashicons-yes"></span></a>';

        }

        $output .= '</span>';

        return $output;

    }


	/**
	 * Add needed hidden fields
	 *
	 * @param string $which
	 *
	 */
	protected function extra_tablenav( $which ) {

		if( $which == 'top' )
			return;

		// Add calendar id as a hidden field
		echo '<input type="hidden" name="calendar_id" value="' . ( ! empty( $_GET['calendar_id'] ) ? absint( $_GET['calendar_id'] ) : 0 ) . '" />';

		// Add a custom nonce for the list table
		wp_nonce_field( 'wpsbc_list_table_legend_items', 'wpsbc_token', false );

	}


	/**
	 * HTML display when there are no items in the table
	 *
	 */
	public function no_items() {

		echo '<div class="wpsbc-list-table-no-items">';
			echo '<p>' . __( 'Oops... it seems there are no legend items', 'wp-simple-booking-calendar' ) . '</p>';
			echo '<a class="button-primary wpsbc-button-large" href="' . add_query_arg( array( 'subpage' => 'add-legend-item' ) ) . '">' . __( 'Set Up A Legend Item', 'wp-simple-booking-calendar' ) . '</a>';
		echo '</div>';

	}

}