<?php

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Validates and handles the resetting of the Calendar's ical_hash
 *
 */
function wpsbc_action_csv_export()
{

    // Verify for nonce
    if (empty($_POST['wpsbc_token']) || !wp_verify_nonce($_POST['wpsbc_token'], 'wpsbc_csv_export')) {
        return;
    }

    // Verify the calendar id
    if (empty($_POST['calendar_id'])) {
        wpsbc_admin_notices()->register_notice('csv_export_calendar_id_missing', '<p>' . __('Invalid Calendar ID.', 'wp-simple-booking-calendar') . '</p>', 'error');
        wpsbc_admin_notices()->display_notice('csv_export_calendar_id_missing');
        return;
    }

    // Get the calendar
    $calendar_id = absint($_POST['calendar_id']);

    // Verify for legend items
    if (empty($_POST['csv-export-legend-items'])) {
        wpsbc_admin_notices()->register_notice('csv_export_legends_missing', '<p>' . __('Please select at least one legend item to export.', 'wp-simple-booking-calendar') . '</p>', 'error');
        wpsbc_admin_notices()->display_notice('csv_export_legends_missing');
        return;
    }

    $selected_legend_items = array();

    // Get legent item names
    foreach ($_POST['csv-export-legend-items'] as $export_legend_item) {
        $legend_item = wpsbc_get_legend_item($export_legend_item);
        $selected_legend_items[$export_legend_item] = $legend_item->get('name');
    }

    // Get events
    $events = wpsbc_get_events(array('calendar_id' => $calendar_id, 'orderby' => 'date_year, date_month, date_day', 'order' => 'ASC'));


    //CSV Header
    if ($_POST['csv-export-format'] == 'groupped_date') {
        $csv_header = array('Date' => '-', 'Legend' => '-', 'Description' => '-');
    } else {
        $csv_header = array('Year' => '-', 'Month' => '-', 'Day' => '-', 'Legend' => '-', 'Description' => '-');
    }

    // Set Start Date
    if(isset($_POST['wpsbc-export-csv-start-date']) && !empty($_POST['wpsbc-export-csv-start-date'])){
        $start_date = DateTime::createFromFormat('Y-m-d', $_POST['wpsbc-export-csv-start-date']);
    }

    // Set End Date
    if(isset($_POST['wpsbc-export-csv-end-date']) && !empty($_POST['wpsbc-export-csv-end-date'])){
        $end_date = DateTime::createFromFormat('Y-m-d', $_POST['wpsbc-export-csv-end-date']);
    }

    $csv_lines = array();

    // Add the CSV header
    foreach ($csv_header as $header_key => $header_value) {
        $csv_lines[0][$header_key] = $header_key;
    }

    $i = 1;

    // Loop through events
    foreach ($events as $event) {

        // Check if legend item is correct
        if (!array_key_exists($event->get('legend_item_id'), $selected_legend_items)) {
            continue;
        }

        $event_date = new DateTime();
        $event_date->setDate($event->get('date_year'), $event->get('date_month'), $event->get('date_day'));

        // Check start date
        if(isset($start_date) && $event_date < $start_date){
            continue;
        }

        // Check end date 
        if(isset($end_date) && $event_date > $end_date){
            continue;
        }

        $csv_lines[$i] = $csv_header;

        // Add event to CSV
        if ($_POST['csv-export-format'] == 'groupped_date') {
            $csv_lines[$i] = array(
                'Date' => $event_date->format(get_option('date_format')),
                'Legend' => $selected_legend_items[$event->get('legend_item_id')],
                'Description' => $event->get('description'),
            );
        } else {
            $csv_lines[$i] = array(
                'Year' => $event_date->format('Y'),
                'Month' => $event_date->format('n'),
                'Day' => $event_date->format('j'),
                'Legend' => $selected_legend_items[$event->get('legend_item_id')],
                'Description' => $event->get('description'),
            );
        }

        $i++;

    }

    if($i === 1){
        wpsbc_admin_notices()->register_notice('csv_export_file_empty', '<p>' . __('No events matched your criteria.', 'wp-simple-booking-calendar') . '</p>', 'error');
        wpsbc_admin_notices()->display_notice('csv_export_file_empty');
        return;
    }

    // Output headers so that the file is downloaded rather than displayed
    header('Content-type: text/csv');
    header('Content-Disposition: attachment; filename="wpsbc-dates-export-calendar-' . $calendar_id . '-' . time() . '.csv"');

    // Do not cache the file
    header('Pragma: no-cache');
    header('Expires: 0');
    echo "\xEF\xBB\xBF"; // UTF-8 BOM

    // Create a file pointer connected to the output stream
    $file = fopen('php://output', 'w');

    // Output each row of the data
    foreach ($csv_lines as $line) {
        fputcsv($file, $line);
    }

    exit();

}
add_action('wpsbc_action_csv_export', 'wpsbc_action_csv_export');
